package STIX::MalwareAnalysis;

use 5.010001;
use strict;
use warnings;
use utf8;

use STIX::Common::List;
use STIX::Common::OpenVocabulary;

use Moo;

use Types::Standard qw(Str Enum InstanceOf);
use Types::TypeTiny qw(ArrayLike);

use namespace::autoclean;

extends 'STIX::Common::Properties';

use constant SCHEMA =>
    'http://raw.githubusercontent.com/oasis-open/cti-stix2-json-schemas/stix2.1/schemas/sdos/malware-analysis.json';

use constant PROPERTIES => (
    qw(type spec_version id created modified),
    qw(created_by_ref revoked labels confidence lang external_references object_marking_refs granular_markings extensions),
    qw(product version host_vm_ref operating_system_ref installed_software_refs configuration_version modules analysis_engine_version analysis_definition_version submitted analysis_started analysis_ended result_name result analysis_sco_refs sample_ref)
);

use constant STIX_OBJECT      => 'SDO';
use constant STIX_OBJECT_TYPE => 'malware-analysis';

has product              => (is => 'rw', isa => Str, required => 1);
has version              => (is => 'rw', isa => Str);
has host_vm_ref          => (is => 'rw', isa => InstanceOf ['STIX::Observable::Software']);
has operating_system_ref => (is => 'rw', isa => InstanceOf ['STIX::Observable::Software']);

has installed_software_refs => (
    is      => 'rw',
    isa     => ArrayLike [InstanceOf ['STIX::Observable::Software']],
    default => sub { STIX::Common::List->new }
);

has configuration_version       => (is => 'rw', isa => Str);
has modules                     => (is => 'rw', isa => ArrayLike [Str], default => sub { STIX::Common::List->new });
has analysis_engine_version     => (is => 'rw', isa => Str);
has analysis_definition_version => (is => 'rw', isa => Str);

has submitted => (
    is     => 'rw',
    isa    => InstanceOf ['STIX::Common::Timestamp'],
    coerce => sub { ref($_[0]) ? $_[0] : STIX::Common::Timestamp->new($_[0]) },
);

has analysis_started => (
    is     => 'rw',
    isa    => InstanceOf ['STIX::Common::Timestamp'],
    coerce => sub { ref($_[0]) ? $_[0] : STIX::Common::Timestamp->new($_[0]) },
);

has analysis_ended => (
    is     => 'rw',
    isa    => InstanceOf ['STIX::Common::Timestamp'],
    coerce => sub { ref($_[0]) ? $_[0] : STIX::Common::Timestamp->new($_[0]) },
);

has result_name => (is => 'rw', isa => Str);
has result      => (is => 'rw', isa => Enum [STIX::Common::OpenVocabulary->MALWARE_RESULT()]);
has analysis_sco_refs =>
    (is => 'rw', isa => ArrayLike [InstanceOf ['STIX::Observable']], default => sub { STIX::Common::List->new });

has sample_ref => (
    is  => 'rw',
    isa => InstanceOf ['STIX::Observable::File', 'STIX::Observable::NetworkTraffic', 'STIX::Observable::Artifact']
);

1;

=encoding utf-8

=head1 NAME

STIX::MalwareAnalysis - STIX Domain Object (SDO) - Malware Analysis

=head1 SYNOPSIS

    use STIX::MalwareAnalysis;

    my $malware_analysis = STIX::MalwareAnalysis->new();


=head1 DESCRIPTION

Malware Analysis captures the metadata and results of a particular analysis
performed (static or dynamic) on the malware instance or family.


=head2 METHODS

L<STIX::MalwareAnalysis> inherits all methods from L<STIX::Common::Properties>
and implements the following new ones.

=over

=item STIX::MalwareAnalysis->new(%properties)

Create a new instance of L<STIX::MalwareAnalysis>.

=item $malware_analysis->analysis_definition_version

The version of the analysis definitions used by the analysis tool.

=item $malware_analysis->analysis_ended

The date and time that the malware analysis ended.

=item $malware_analysis->analysis_engine_version

The version of the analysis engine or product that was used to perform this
analysis.

=item $malware_analysis->analysis_sco_refs

The list of STIX objects that were captured during the analysis process.

=item $malware_analysis->analysis_started

The date and time that the malware analysis was initiated.

=item $malware_analysis->configuration_version

The version of the analysis product configuration that was used to perform
this analysis.

=item $malware_analysis->host_vm_ref

A description of the virtual machine environment used to host the guest
operating system (if applicable) that was used for the dynamic analysis of
the malware instance or family.

=item $malware_analysis->id

=item $malware_analysis->installed_software_refs

Any non-standard software installed on the operating system used for the
dynamic analysis of the malware instance or family.

=item $malware_analysis->modules

The particular analysis product modules that were used to perform the
analysis.

=item $malware_analysis->operating_system_ref

The operating system that was used to perform the dynamic analysis.

=item $malware_analysis->product

The name of the analysis engine or product that was used for this analysis.

=item $malware_analysis->result

The classification result as determined by the scanner or tool analysis
process.

=item $malware_analysis->result_name

The classification result or name assigned to the malware instance by the
scanner tool.

=item $malware_analysis->sample_ref

Refers to the object this analysis was performed against.

=item $malware_analysis->submitted

The date and time that this malware was first submitted for scanning or
analysis.

=item $malware_analysis->type

The type of this object, which MUST be the literal C<malware-analysis>.

=item $malware_analysis->version

The version of the analysis product that was used to perform this analysis.

=back


=head2 HELPERS

=over

=item $malware_analysis->TO_JSON

Encode the object in JSON.

=item $malware_analysis->to_hash

Return the object HASH.

=item $malware_analysis->to_string

Encode the object in JSON.

=item $malware_analysis->validate

Validate the object using JSON Schema (see L<STIX::Schema>).

=back


=head1 SUPPORT

=head2 Bugs / Feature Requests

Please report any bugs or feature requests through the issue tracker
at L<https://github.com/giterlizzi/perl-STIX/issues>.
You will be notified automatically of any progress on your issue.

=head2 Source Code

This is open source software.  The code repository is available for
public review and contribution under the terms of the license.

L<https://github.com/giterlizzi/perl-STIX>

    git clone https://github.com/giterlizzi/perl-STIX.git


=head1 AUTHOR

=over 4

=item * Giuseppe Di Terlizzi <gdt@cpan.org>

=back


=head1 LICENSE AND COPYRIGHT

This software is copyright (c) 2024 by Giuseppe Di Terlizzi.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
