/*                                                                
**  Copyright (C) 1996,2007,2010  Smithsonian Astrophysical Observatory 
*/                                                                

/*                                                                          */
/*  This program is free software; you can redistribute it and/or modify    */
/*  it under the terms of the GNU General Public License as published by    */
/*  the Free Software Foundation; either version 3 of the License, or       */
/*  (at your option) any later version.                                     */
/*                                                                          */
/*  This program is distributed in the hope that it will be useful,         */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/*  GNU General Public License for more details.                            */
/*                                                                          */
/*  You should have received a copy of the GNU General Public License along */
/*  with this program; if not, write to the Free Software Foundation, Inc., */
/*  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.             */
/*                                                                          */

/* Creation routines.
**/

#ifdef __linux__
/* stdio.h: popen, pclose */
#define _POSIX_C_SOURCE 2
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "pfile.h"
#include "ptemplat.h"
#include "parameter.h"
#include "range.h"
#include "find.h"
#include "putil.h"

#include "param_get_via_xpa.h"

static char *PParseFile (char *value, char *fname, char *pname);

Parameter *
ParamAlloc (void)
{
  Parameter *p;

  p = (Parameter *) malloc (sizeof (Parameter));

  p->pname = NULL;
  p->ptype = 0;
  p->phook = NULL;
  p->pmode = 0;
  p->pquery = 0;
  p->redirect = 0;
  p->pvalue.value = NULL;
  p->pvalue.indir = NULL;
  p->pvalue.type = 0;
  p->pmin.value = NULL;
  p->pmin.indir = NULL;
  p->pmin.type = 0;
  p->pmax.value = NULL;
  p->pmax.indir = NULL;
  p->pmax.type = 0;
  p->pprompt = NULL;
  p->pdefault = NULL;
  p->pflag = 0;
  return p;
}


Parameter *
ParamName (Parameter *p, char *name)
{
  if (p == NULL)
    p = ParamAlloc ();

  p->pname = name;

  return p;
}



Parameter *
ParamType (Parameter *p, VType type)
{
  if (type == 0)
    return NULL;
  if (p == NULL)
    p = ParamAlloc ();

  p->ptype = type;

  return p;
}



Parameter *
ParamMode (Parameter *p, int *mode)
{
  if (mode == NULL)
    return NULL;
  if (p == NULL)
    p = ParamAlloc ();

  p->pmode = *mode;

  return p;
}



Parameter *
ParamMin (Parameter *p, VType type, void *min)
{
  if (p == NULL)
    p = ParamAlloc ();

  p->pmin.type = type;
  p->pmin.value = min;

  return p;
}



Parameter *
ParamMax (Parameter *p, VType type, char *max)
{
  if (p == NULL)
    p = ParamAlloc ();

  p->pmax.type = type;
  p->pmax.value = max;

  return p;
}



Parameter *
ParamValue (Parameter *p, VType type, char *value)
{
  if (p == NULL)
    p = ParamAlloc ();

  p->pvalue.type = type;
  p->pvalue.value = value;

  return p;
}

#define STRINGIFY(arg) #arg

/* routine to store the default parameter value as a text string */
Parameter *
ParamDefault (Parameter *p)
{
  if (p)
    {
      int length;
      char valustr[SZ_PFLINE] = "";

      ParamEntry (NULL, p, NULL, NULL, NULL, valustr, NULL, NULL, NULL);

      length = strlen (valustr);

      if ( length > SZ_PFLINE ) {
	parerr = PARARGS;
	paramerr (1, "Input string is longer than " STRINGIFY(SZ_PFLINE) " chars", p->pname );
	return NULL;
      }

      p->pdefault = (char *) malloc (length + 1);
      strcpy (p->pdefault, valustr);
    }
  return p;
}

Parameter *
ParamPrompt (Parameter *p, char *prompt)
{
  if (p == NULL)
    p = ParamAlloc ();

  p->pprompt = prompt;

  return p;
}


Parameter *
ParamStruct (Parameter *param, char *line)
{
  param->pvalue.value = line;

  return param;
}


int 
paramvalid (
    ParamFile pfile,
    char *pname,
    char *pvalue,
    char *cvalue		/* Cannonical value             */
)
{
  if (pfile != NULL)
    {
      Parameter *param;
      Value value;

      value.type = StringType;
      value.value = CXCNewString (pvalue);
      value.indir = NULL;

      if ((param = ParamLookup (pfile->psets, pname)) != NULL)
	{
	  Value *check;
	  if ((check = cxcValidate (pfile, param, &value, 0)) != NULL)
	    {
	      if (cvalue)
		{
		  Value tvalue;
		  tvalue.type = StringType;
		  tvalue.value = pvalue;
		  tvalue.indir = NULL;

		  if (VIsIndirect (&tvalue))
		    strcpy (cvalue, pvalue);
		  else
		    Convert (check->value, check->type, cvalue, StringType);

		  if (check != &value)
		    free (check);
		}
	      free (value.value);
	      return 1;
	    }
	}
      free (value.value);
    }
  return 0;
}


/* Methods */

int 
VXSize (void *value)
{
  return sizeof (int);
}


Value *
cxcValidate (ParamFile pfile, Parameter *p, Value *value, int interactive)
{

  if (p == NULL)
    return NULL;

#ifndef NONCIAO
  if (interactive == 1)
    {
      value_str = (char *) value->value;

      if (value_str != NULL)
	{
	  if (*value_str == '?')
	    {

	      if (paramHelpFunction (paramToolName, p->pname, ++value_str))
		{
		  fprintf (stderr, "No help available for %s.%s or %s\n",
			   paramToolName, p->pname, ++value_str);
		}


	      return (NULL);
	    }
	}
    }
#endif /* NONCIAO */

  if ((p->ptype & ListType) && !IsInList (p))
    {
      if (!VAccess (pfile, p, (FileType | ListType), value))
	return NULL;
      else
	return value;
    }
  if (!VAccess (pfile, p, p->ptype, value))
    return NULL;

  if (!VRange (pfile, p, &p->pmin))
    return PRangeMinMax (value, VAccess (pfile, p, p->ptype, &p->pmin),
			 VAccess (pfile, p, p->ptype, &p->pmax));

  if (p->pmax.value)
    {
      parerr = PARRANGEMAX;
      return NULL;
    }

  if (p->pmin.value == NULL)
    {
      parerr = PARBADRANGE;
      return NULL;
    }

  return VInRange (p->pmin.value, value);
}

Value *
VXAcce (ParamFile pfile, Parameter *p, VType type, Value *value)
{
  void *converted;

  if (type == 0)
    return NULL;
  if (value == NULL)
    return NULL;

  VDirect (pfile, p, value);
  if (value->value == NULL)
    return NULL;

  if (p->ptype & ListType && p->phook == NULL)
    {
      if (TypeIndex (value->type) != StringType
	  && !cxcAccess (value->value, "r"))
	{
	  parerr = PARNOLISTFILE;
	  return NULL;
	}
      else
	{
	  p->phook = NextListValue;
	  p->extension = (listrecord *) malloc (sizeof (listrecord));

	  if ((((listrecord *) p->extension)->file
	       = fopen (value->value, "r")) == NULL)
	    {
	      p->phook = NULL;
	      free (p->extension);
	      parerr = PARNOLISTFILE;

	      return NULL;
	    }
	}
    }

  if (value->type == type)
    return value;

  if ((converted = VConvert (value, NULL, type)) == NULL)
    return NULL;

  return VNewValue (value, converted, type);
}


Value *
VNewValue (Value *pvalue, void *value, VType type)
{
  void *tmp;
  int size;

  if (pvalue == NULL)
    return NULL;
  if (value == NULL)
    return NULL;
  if (type == 0)
    return NULL;

  size = VSize (type, value);
  tmp = (char *) malloc (size);
  memmove (tmp, value, size);

  if (pvalue->value)
    free (pvalue->value);

  pvalue->type = type;
  pvalue->value = tmp;

  return pvalue;
}


Value *
VIsIndirect (Value *value)
{
  if (value && value->indir)
    return value;
  if (value && value->value && TypeIndex (value->type) == StringType)
    {
      if ((*(char *) value->value == ')') ||
	  (strstr ((char *) value->value, "${") != NULL) ||
	  (strstr ((char *) value->value, "%xpa(") != NULL))
	{
	  return value;
	}
      else
	return NULL;
    }
  else
    return NULL;
}


void 
VDirect (ParamFile pfile, Parameter *param, Value *value)
{
  char fname[SZ_PFLINE];
  char pname[SZ_PFLINE];
  int mode = HMODE | (pfile->mode & HHMODE);
  void *converted = NULL;
  char *indirect;

  char class[1024];

  /* check for indirect classing and change value if necessary */
  IndirectClassing (pfile, param, value, class);

  if (!VIsIndirect (value))
    return;

  if (value->indir)
    {
      if (value->value)
	free (value->value);
      value->type = StringType;
      value->value = value->indir;
      value->indir = NULL;
    }

  /* look for )) -- execute command and return output */
  indirect = (char *) value->value;

  if (*(indirect + 1) == ')')
    {
      int got;
      char *s;
      char buf[SZ_PFLINE * 2];
      FILE *fd;

      memset (buf, 0, SZ_PFLINE * 2);

      value->indir = value->value;
      value->value = NULL;

      indirect += 2;
      if ((fd = popen (indirect, "r")) == NULL)
	{
	  parerr = PARBADINDIRECT;
	  return;
	}

      got = 0;
      *buf = '\0';
      while ((s = fgets (&buf[got], SZ_PFLINE, fd)))
	{
	  got += strlen (s);
	  if (got < 1)
	    got = 1;		/* avoid array bounds issue */
	  if (buf[got - 1] == '\n')
	    buf[got - 1] = ' ';
	  if (got > SZ_PFLINE)
	    {
	      buf[SZ_PFLINE] = '\0';
	      got = SZ_PFLINE;
	      break;
	    }
	}
      if (got < 1)
	got = 1;		/* avoid array bounds issue */

      if ((buf[got - 1] == '\n') || (buf[got - 1] == ' '))
	buf[got - 1] = '\0';
      pclose (fd);
      VNewValue (value, buf, StringType);
      return;
    }				/* end  if )) shell command */
  else if (strstr (indirect, "${"))
    {
      char *s;
      char *l;
      char buf[SZ_PFLINE * 2];

      value->indir = value->value;
      value->value = NULL;

      memset (&buf[0], 0, SZ_PFLINE * 2);
      l = indirect;
      while (NULL != (s = strstr (l, "${")))
	{
	  char envv[SZ_PFLINE];
	  char *res;
	  memset (&envv[0], 0, SZ_PFLINE);

	  if (s != l)
	    strncat (buf, l, s - l);

	  if ((s != indirect) && (*(s - 1) == '\\'))
	    {
	      strcpy (buf, "$");
	      l = s + 1;
	      continue;
	    }


	  s += 1;
	  if (*s != '{')
	    {
	      l++;
	      continue;
	    }
	  s += 1;
	  if (strchr (l, '}') == NULL)
	    {
	      l++;
	      continue;
	    }
	  l = strchr (l, '}');
	  strncpy (envv, s, l - s);
	  res = getenv (envv);
	  if (res != NULL)
	    strcat (buf, res);
	  l++;
	}

      strcat (buf, l);

      VNewValue (value, buf, StringType);
      return;

    }
  else if (strstr (indirect, "%xpa("))
    {
      char *s;
      char *l;
      char buf[SZ_PFLINE * 2];

      value->indir = value->value;
      value->value = NULL;

      memset (&buf[0], 0, SZ_PFLINE * 2);
      l = indirect;
      while (NULL != (s = strstr (l, "%xpa(")))
	{
	  char *res;
	  char *cc;
	  char tool[SZ_PFLINE];
	  char cmd[SZ_PFLINE];


	  if (s != l)
	    strncat (buf, l, s - l);

	  if ((s != indirect) && (*(s - 1) == '\\'))
	    {
	      strcpy (buf, "%");
	      l = s + 1;
	      continue;
	    }




	  memset (&tool[0], 0, SZ_PFLINE * sizeof (char));
	  memset (&cmd[0], 0, SZ_PFLINE * sizeof (char));
	  s = strchr (s, '(');
	  cc = strchr (s, ',');
	  if ((s == NULL) || (cc == NULL) || (strchr (s, ')') == NULL))
	    {
	      l++;
	      continue;
	    }
	  l = strchr (s, ')');
	  s += 1;
	  strncpy (tool, s, cc - s);
	  cc += 1;
	  strncpy (cmd, cc, l - cc);

	  res = param_get_via_xpa (tool, cmd);

	  cc = res;
	  if (res)
	    {
	      while (isspace (*(cc++)))
		{
		};
	      strcat (buf, --cc);
	    }
	  free (res);


	  l += 1;

	}			/* end while */

      strcat (buf, l);

      VNewValue (value, buf, StringType);
      return;
    }


  if (param->redirect == 1)
    {
      parerr = PARCYCLICERR;
      if ((value->type == StringType) && VIsIndirect (value) &&
	  param->ptype == StringType)
	{
	  value->indir = value->value;
	  value->value = NULL;
	}
      return;
    }
  else if (strcmp (&indirect[1], param->pname) == 0)
    {
      parerr = PARSELFDIRECT;
      if ((value->type == StringType) && VIsIndirect (value) &&
	  param->ptype == StringType)
	{
	  value->indir = value->value;
	  value->value = NULL;
	}
      return;
    }
  else
    {
      param->redirect = 1;
    }

  if (PParseFile (value->value, fname, pname))
    {

      value->indir = value->value;
      value->value = NULL;

      if (*fname == '\0')	/* Local indirection    */
	if (strcmp (pname, param->pname))
	  {
	    converted = ParamGetX (pfile, pname, value->type, &mode);
	  }
	else
	  {
	    parerr = PARBADINDIRECT;
	    return;
	  }
      else
	{
	  if ((pfile = ParamFileLookup (c_paramfind (fname, "r",
						     PARAMEXTN, PARAMPATH))))
	    {
	      converted = ParamGetX (pfile, pname, value->type, &mode);
	    }
	  else if (NULL !=
		   ((pfile = (ParamFile) paramopen (fname, NULL, 0, "rH"))))
	    {
	      converted = ParamGetX (pfile, pname, value->type, &mode);
	      paramclose (pfile);
	    }
	}

      if (param->redirect)
	{
	  if (converted)
	    VNewValue (value, converted, value->type);
	  else
	    parerr = PARBADINDIRECT;
	  param->redirect = 0;	/* converted val so reset redirect flag */
	}
      return;
    }

  parerr = PARNOFILE;
  free (value->value);
  return;
}


static char *
PParseFile (char *value, char *fname, char *pname)
{
  char *pn;
  char *ext = NULL;

  if (value == NULL)
    return NULL;
  if ((*value != ')') &&
      (strstr (value, "${") != NULL) && (strstr (value, "%xpa(") != NULL))
    return NULL;
  if (*value == ')')
    value++;
  if (*value == '*')
    value++;

  strcpy (fname, value);

  if (NULL != (pn = strrchr (fname, '.')))
    {
      if (!strcmp (pn, ".p_mode") ||
	  !strcmp (pn, ".p_value") ||
	  !strcmp (pn, ".p_min") ||
	  !strcmp (pn, ".p_max") || !strcmp (pn, ".p_prompt"))
	{
	  ext = pn + 1;
	  *pn = '\0';
	  pn = strrchr (fname, '.');
	}
      if (pn)
	{
	  *pn++ = '\0';
	  strcpy (pname, pn);
	}
      else
	{
	  strcpy (pname, fname);
	  *fname = '\0';
	}
      if (ext)
	{
	  strcat (pname, ".");
	  strcat (pname, ext);
	}
    }
  else
    {
      strcpy (pname, fname);
      *fname = '\0';
    }

  return fname;
}

/* conversion buffer.  it's double instead of char to avoid alignment
   problems if a double gets dump at the start */
static double PConverterBuffer[SZ_PFLINE / sizeof (double) + 1];
/* static char PConverterBuffer[SZ_PFLINE]; */

void *
Convert (void *value, VType from, void *copy, VType to)
{
  Converter *conv;

  if ((value == 0) || (to == 0))
    {
      if (copy)
	*((char *) copy) = '\0';
      return 0;
    }

  if (copy == NULL)
    copy = PConverterBuffer;

  if (TypeIndex (from) == TypeIndex (to))
    {
      memmove (copy, value, VSize (from, value));
      return copy;
    }
  else
    for (conv = VTypeList[TypeIndex (to)]->converters; conv->from; conv++)
      if (TypeIndex (conv->from) == TypeIndex (from))
	return (*conv->convert) (value, copy);

  parerr = PARNOCONVERTER;
  return NULL;
}


void *
VConvert (Value *value, void *copy, VType to)
{
  if (value == NULL)
    return NULL;

  return Convert (value->value, value->type, copy, to);
}
